import 'package:flutter/material.dart';
import 'package:flutter_svg/flutter_svg.dart';
import 'package:get/get.dart';
import 'package:viser_remit/core/helper/date_converter.dart';
import 'package:viser_remit/core/helper/string_format_helper.dart';
import 'package:viser_remit/core/utils/my_images.dart';
import 'package:viser_remit/core/utils/util.dart';
import 'package:viser_remit/data/controller/transection/transection_controller.dart';
import 'package:viser_remit/data/repo/transaction_history_repo/transaction_history_repo.dart';
import 'package:viser_remit/view/components/custom_loader/custom_loader.dart';
import 'package:viser_remit/view/components/custom_no_data_found_class.dart';
import 'package:viser_remit/view/screens/transection/widget/filter_section.dart';
import 'package:viser_remit/view/screens/transection/widget/transection_details_bottom_sheet.dart';

import '../../../core/utils/dimensions.dart';
import '../../../core/utils/my_color.dart';
import '../../../core/utils/my_strings.dart';
import '../../../core/utils/style.dart';
import '../../../data/services/api_client.dart';
import '../../components/app-bar/viser_remit_app_bar.dart';
import '../../components/bottom-sheet/custom_bottom_sheet.dart';
import '../../components/bottom-sheet/custom_bottom_sheet_plus.dart';

class TransactionScreen extends StatefulWidget {
  const TransactionScreen({super.key});

  @override
  State<TransactionScreen> createState() => _TransactionScreenState();
}

class _TransactionScreenState extends State<TransactionScreen> {
  bool leadingStatus = false;

  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(TransactionHistoryRepo(apiClient: Get.find()));
    final controller = Get.put(TransactionController(transactionHistoryRepo: Get.find()));

    leadingStatus = Get.arguments ?? false;

    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.loadTransactionHistory();
    });
  }

  @override
  Widget build(BuildContext context) {
    var apiClient = Get.find<ApiClient>();

    return GetBuilder<TransactionController>(
        builder: (controller) => Scaffold(
              appBar: ViserRemitAppBar(
                title: MyStrings.transaction,
                // appBarHeight: 65,
                isElevation: true,
                isShowLeading: leadingStatus,
              ),
              body: controller.isLoading
                  ? const CustomLoader()
                  : SingleChildScrollView(
                      physics: const BouncingScrollPhysics(),
                      padding: const EdgeInsets.symmetric(horizontal: Dimensions.space14),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          const SizedBox(height: Dimensions.space18),
                          controller.transactionDataList.isEmpty
                              ? const NoDataOrInternetScreen()
                              : Column(
                                  children: List.generate(controller.transactionDataList.length, (index) {
                                    var transactionData = controller.transactionDataList[index];
                                    return GestureDetector(
                                      onTap: () {
                                        CustomBottomSheet(
                                          child: TransectionDetailsBottomSheet(controller: controller, transactionData: transactionData),
                                        ).customBottomSheet(context);
                                      },
                                      child: Container(
                                        padding: const EdgeInsets.symmetric(horizontal: 13, vertical: 16),
                                        margin: const EdgeInsetsDirectional.only(bottom: 8),
                                        decoration: BoxDecoration(
                                          borderRadius: BorderRadius.circular(4),
                                          border: const Border(left: BorderSide(color: MyColor.primaryColor,width: 1.5)),
                                          // boxShadow: MyUtils.getShadow(),
                                          color: MyColor.colorWhite,
                                          boxShadow: MyUtils.getCardShadow()
                                        ),
                                        child: Row(
                                          children: [
                                            Expanded(
                                              flex: 5,
                                              child: Column(
                                                crossAxisAlignment: CrossAxisAlignment.start,
                                                children: [
                                                  Row(
                                                    children: [
                                                      Text(MyStrings.trxNumber.tr, style: semiBoldDefault),
                                                      const SizedBox(width: 4,),
                                                      Expanded(child: Text("#${transactionData.trx}", style: boldDefault.copyWith(color: MyColor.bodyTextColor))),
                                                    ],
                                                  ),
                                                  const SizedBox(height: Dimensions.space10,),
                                                  Text(MyStrings.amount.tr, style: semiBoldLarge),

                                                  Text(
                                                    "${transactionData.trxType.toString()}"
                                                        "${Converter.formatNumber(transactionData.amount.toString())} "
                                                        "${apiClient.getCurrencyOrUsername(isCurrency: true)}",
                                                    maxLines: 2,
                                                    style: boldSmall.copyWith(color: transactionData.trxType == '+' ? MyColor.greenP : MyColor.colorRed),
                                                  ),
                                                ],
                                              ),
                                            ),
                                            // const SizedBox(width: Dimensions.space24),
                                            Expanded(
                                              flex: 2,
                                              child: Align(
                                                alignment: Alignment.centerRight,
                                                child: Column(
                                                  crossAxisAlignment: CrossAxisAlignment.start,
                                                  children: [
                                                    Text(
                                                      DateConverter.isoStringToLocalDateOnly(transactionData.createdAt.toString()),
                                                      style: mediumDefault.copyWith(color: MyColor.bodyTextColor),
                                                    ),

                                                    const SizedBox(height: Dimensions.space10),
                                                    const Text(MyStrings.charge, style: boldLarge),
                                                    Text("${Converter.formatNumber(transactionData.charge.toString())} ${apiClient.getCurrencyOrUsername(isCurrency: true)}", style: mediumDefault),

                                                  ],
                                                ),
                                              ),
                                            ),
                                          ],
                                        ),
                                      ),
                                    );
                                  }),
                                ),
                        ],
                      ),
                    ),
              floatingActionButton: Padding(
                padding: const EdgeInsetsDirectional.only(bottom: 20),
                child: FloatingActionButton(
                  backgroundColor: MyColor.colorWhite,
                  onPressed: () {
                    CustomBottomSheetPlus(
                      child: GetBuilder<TransactionController>(
                        builder: (controller) => FilterSection(controller: controller),
                      ),
                    ).show(context);
                  },
                  child: SvgPicture.asset(
                    MyImages.floatingIcon,
                    colorFilter: ColorFilter.mode(controller.filterSearch ? MyColor.primaryColor : MyColor.bodyTextColor, BlendMode.srcIn),
                  ),
                ),
              ),
            ));
  }
}
